'This object handles the chess board
'Copyright 2008 Raymond Allen

'0  1  2  3  4  5  6  7
'8  9  10 11 12 13 14 15
'16 17 18 19 20 21 22 23
'24 25 26 27 28 29 30 31
'32 33 34 35 36 37 38 39
'40 41 42 43 44 45 46 47
'48 49 50 51 52 53 54 55
'56 57 58 59 60 61 62 63

CON
    'number of custom 16x16 characters              
  nuchars = 13*9+8-9    '!!!!! you must have the correct # here for alignment later 'Using tile driver space for blank square

  'Piece Colors
  White=0  'Because White goes first
  Black=1
  nColors=2
  '  // Codes representing pieces
  PAWN = 0
  KNIGHT = 2
  BISHOP = 4
  ROOK = 6
  QUEEN = 8
  KING = 10
  AllPieces = 12'16    'Note:  "All" does not include EnPassant or ExtraKing
  NoPiece = 14'18
  'nRegularPieces = 12 
  WHITE_PAWN = PAWN + WHITE
  WHITE_KNIGHT = KNIGHT + WHITE
  WHITE_BISHOP = BISHOP + WHITE
  WHITE_ROOK = ROOK + WHITE
  WHITE_QUEEN = QUEEN + WHITE
  WHITE_KING = KING + WHITE
  White_AllPieces = AllPieces + White
  BLACK_PAWN = PAWN + BLACK
  BLACK_KNIGHT = KNIGHT + BLACK
  BLACK_BISHOP = BISHOP + BLACK
  BLACK_ROOK = ROOK + BLACK
  BLACK_QUEEN = QUEEN + BLACK
  BLACK_KING = KING + BLACK
  Black_AllPieces = AllPieces + Black
  'bitboard constants
  nBitBoards=14
  'Move constants 
  'Move Structure (byte withing long)
  MOVE_SourceSquare=0
  MOVE_DestinationSquare=1
  MOVE_MovedPiece=2
  MOVE_Type=3
  MOVE_CapturedPiece=4
  MOVE_Bytes=5  '# of bytes in move structure

  nMaxMoves=250  'wikipedia says 218 max, but some here may not be valid
  MoveListSize=MOVE_Bytes* nMaxMoves
  
  '// The different types of moves recognized by the game (one byte)
  MOVE_NORMAL = 0
  MOVE_CAPTURE = 1
  MOVE_EN_PASSANT = 2
  MOVE_CASTLING_KINGSIDE = 4
  MOVE_CASTLING_QUEENSIDE = 8
  MOVE_RESIGN = 15
  MOVE_STALEMATE = 14
  MOVE_PROMOTION_KNIGHT = 16
  MOVE_PROMOTION_BISHOP = 32
  MOVE_PROMOTION_ROOK = 64
  MOVE_PROMOTION_QUEEN = 128
  '// A pair of masks used to split the promotion and the non-promotion part of a move type ID
  PROMOTION_MASK = %1111_0000
  NO_PROMOTION_MASK = %0000_1111
  'Castling masks
  EmptySquares_White_Kingside=(%0110_0000)<<24
  EmptySquares_White_Queenside=(%0000_1100)<<24
  EmptySquares_Black_Kingside=%0110_0000
  EmptySquares_Black_Queenside=%0000_1100
  Extrakings_White_Kingside = (%0011_0000)<<24 
  Extrakings_White_Queenside = (%0001_1000)<<24     
  Extrakings_Black_Kingside = %0011_0000
  Extrakings_Black_Queenside = %0001_1000

  'Memory Settings  
  GameStateSize=1+nBitBoards<<1 +2

  'AlphaBeta constants
  ' Search node types: MAXNODEs are nodes where the computer player is the
  ' one to move  MINNODEs are positions where the opponent is to move.
  MAXNODE = true 
  MINNODE = false 
  ' Alphabeta search boundaries
  ALPHABETA_MAXVAL = 30000 
  ALPHABETA_MINVAL = -30000 
  ALPHABETA_ILLEGAL = -31000 
  ' An approximate upper bound on the total value of all positional
  ' terms in the evaluation function
  EVAL_THRESHOLD = 200 
  ' A score below which we give up: if Alphabeta ever returns a value lower
  ' than this threshold, then all is lost and we might as well resign.  Here,
  ' the value is equivalent to "mated by the opponent in 3 moves or less".
  ALPHABETA_GIVEUP = -29995

  AlphaBetaDepth=1

'0  1  2  3  4  5  6  7
'8  9  10 11 12 13 14 15
'16 17 18 19 20 21 22 23
'24 25 26 27 28 29 30 31
'32 33 34 35 36 37 38 39
'40 41 42 43 44 45 46 47
'48 49 50 51 52 53 54 55
'56 57 58 59 60 61 62 63  
  
OBJ
  ser:  "fullduplexserial"  'for debugging

VAR
  'for custom characters
  word user_charbase
  'bitboards:  'need 2 longs for each (64-bit) bitboard 
  long GameState 'byte[0]=Turn (1=black,0=white), byte[1]=bWhiteCanCastle (bit0=bCanCastleQueenside, bit1=bCanCastleKingside), byte[2]=bBlackCanCastle(bit0=bCanCastleQueenside, bit1=bCanCastleKingside), byte[3]=EnPassant Square (if any)
  long BitBoard[nBitBoards<<1]  'The entire state of the game is contained within this array, the previous long, and next two longs
  long MaterialValue[2]
  long GameStateBackup[GameStateSize]
  long AlphaBetaGameStateBackup[GameStateSize*AlphaBetaDepth]
  
  long SquareBitBoard[2]  'used for converting square to bitboard  
  'game vars
  Byte Player  'Whose turn it is (from GameState.byte[0]
  
  byte NumPawns[nColors]
  'move vars
  byte CurrentMove[MOVE_Bytes]

  'list of legal moves
  long nLegalMoves
  byte LegalMoveList[MoveListSize]
  byte AlphaBetaLegalMoveList[MoveListSize*AlphaBetaDepth] 
  byte LegalMoveListBackup[MoveListSize]

  byte CapturedPiece 'for calls to AddToMoveList...

  long debug

  'Evaluation vars
  byte MaxPawnFileBins [ 8 ]
  byte MaxPawnColorBins [ 2 ]
  byte MaxMostAdvanced [ 8 ]
  byte MaxPassedPawns [ 8 ]
  byte MinPawnFileBins [ 8 ]
  byte MinMostBackward [ 8 ]      
  byte MaxTotalPawns
  byte PawnRams


PUB Start|i
  '64 byte align the user characters
  user_charbase := @uchar & $FFC0 'destination
  'user_charbase_offset := user_charbase-@uchar  
  longmove(user_charbase,@uchar,16*nuchars)

  'Clear space in tile driver to reuse as blank square bitmap
  repeat i from 0 to 9*16
    long[pBlank][i]:=0

  'start debug term
  ser.start(31,30,0,9600)  

PRI EmptyBoard 'clear all bitboards
  longfill(@BitBoard,0,nBitBoards<<1)

Pub Init:bOK|i
  'Setup the board to play
   'clear running totals
  repeat i from white to black
    MaterialValue[i]:=0
    NumPawns[i]:=0

   'Put the pieces on the board
  EmptyBoard
  AddPiece( 0, BLACK_ROOK )
  AddPiece( 1, BLACK_KNIGHT )
  AddPiece( 2, BLACK_BISHOP )
  AddPiece( 3, BLACK_QUEEN )
  AddPiece( 4, BLACK_KING )
  AddPiece( 5, BLACK_BISHOP )
  AddPiece( 6, BLACK_KNIGHT )
  AddPiece( 7, BLACK_ROOK )
  repeat  i from 8 to 15
    AddPiece( i, BLACK_PAWN )
  'repeat i from 3 to 5
  '  AddPiece(i, Black_ExtraKing)

  repeat i from 48 to 55
    AddPiece( i, WHITE_PAWN )
  AddPiece( 56, WHITE_ROOK )
  AddPiece( 57, WHITE_KNIGHT )
  AddPiece( 58, WHITE_BISHOP )
  AddPiece( 59, WHITE_QUEEN )
  AddPiece( 60, WHITE_KING )
  AddPiece( 61, WHITE_BISHOP )
  AddPiece( 62, WHITE_KNIGHT )
  AddPiece( 63, WHITE_ROOK )
  'repeat i from 59 to 61
  '  AddPiece(i, White_ExtraKing)

  GameState.byte[0]:=0 'White's Turn
  GameState.byte[1]:=3 'White can castle both ways
  GameState.byte[2]:=3 'Black can castle both ways
  GameState.byte[3]:=-1 'No EnPassant Square
   'White to play the first move
  Player:=White
  return true  

PRI EvaluateComplete:value 'Evaluate Board
  value:=EvalMaterial
  value+=EvalPawnStructure
  ser.dec(value)
  ser.tx(13)



PRI  EvalPawnStructure:score|bin,i,amax,amin
  '// Given the pawn formations, penalize or bonify the position according to
  '// the features it contains
    score:= 0

    '// First, look for doubled pawns
    '// In chess, two or more pawns on the same file usually hinder each other,
    '// so we assign a minor penalty
    repeat  bin from 0 to 7
      if ( MaxPawnFileBins[ bin ] > 1 )
        score -= 8

    '// Now, look for an isolated pawn, i.e., one which has no neighbor pawns
    '// capable of protecting it from attack at some point in the future
    if ( ( MaxPawnFileBins[ 0 ] > 0 ) and ( MaxPawnFileBins[ 1 ] == 0 ) )
      score -= 15
    if ( ( MaxPawnFileBins[ 7 ] > 0 ) and ( MaxPawnFileBins[ 6 ] == 0 ) )
      score -= 15
    repeat bin from 1 to 6
      if ( ( MaxPawnFileBins[ bin ] > 0 ) and ( MaxPawnFileBins[ bin - 1 ] == 0 )and ( MaxPawnFileBins[ bin + 1 ] == 0 ) )
        score -= 15

    '// Assign a small penalty to positions in which Max still has all of his
    '// pawns; this incites a single pawn trade (to open a file), but not by
    '// much
    if ( MaxTotalPawns == 8 )
      score -= 10

    '// Penalize pawn rams, because they restrict movement
    score -= 8 * PawnRams

    '// Finally, look for a passed pawn; i.e., a pawn which can no longer be
    '// blocked or attacked by a rival pawn
    if ( player==WHITE )
      amin:=MinMostBackward[ 0 ] <# MinMostBackward[ 1 ]
      if ( MaxMostAdvanced[ 0 ] < amin )
        score += ( 8 - ( MaxMostAdvanced[ 0 ] >> 3 ) ) * ( 8 - ( MaxMostAdvanced[ 0 ] >> 3 ) )
      amin:=MinMostBackward[ 7 ] <# MinMostBackward[ 6 ]        
      if ( MaxMostAdvanced[ 7 ] < amin )
        score += ( 8 - ( MaxMostAdvanced[ 7 ] >> 3 ) ) *  ( 8 - ( MaxMostAdvanced[ 7 ] >> 3 ) )
      repeat i from 1 to 6
      
        if ( ( MaxMostAdvanced[ i ] < MinMostBackward[ i ] ) and  ( MaxMostAdvanced[ i ] < MinMostBackward[ i - 1 ] ) and  ( MaxMostAdvanced[ i ] < MinMostBackward[ i + 1 ] ) )
          score += ( 8 - ( MaxMostAdvanced[ i ] >> 3 ) ) *    ( 8 - ( MaxMostAdvanced[ i ] >> 3 ) )
    else '// from Black's perspective
      amax:= MinMostBackward[ 0]  #> MinMostBackward[ 1 ] 
      if ( MaxMostAdvanced[ 0 ] > amax )
        score += ( MaxMostAdvanced[ 0 ] >> 3 ) *  ( MaxMostAdvanced[ 0 ] >> 3 )
      amax:= MinMostBackward[ 7]  #> MinMostBackward[ 6 ]  
      if ( MaxMostAdvanced[ 7 ] > amax )
        score += ( MaxMostAdvanced[ 7 ] >> 3 ) * ( MaxMostAdvanced[ 7 ] >> 3 )
      repeat i from 1 to 6
        if ( ( MaxMostAdvanced[ i ] > MinMostBackward[ i ] ) and ( MaxMostAdvanced[ i ] > MinMostBackward[ i - 1 ] ) and  ( MaxMostAdvanced[ i ] > MinMostBackward[ i + 1 ] ) )
          score += ( MaxMostAdvanced[ i ] >> 3 ) * ( MaxMostAdvanced[ i ] >> 3 )

    return score



PRI  AnalyzePawnStructure|i,square,rank,afile
  '// Look at pawn positions to be able to detect features such as doubled,
  '// isolated or passed pawns
    '// Reset the counters
    repeat i from 0 to 7
      MaxPawnFileBins[ i ]:= 0
      MinPawnFileBins[ i ]:= 0
    
    MaxPawnColorBins[ 0 ]:= 0
    MaxPawnColorBins[ 1 ]:= 0
    PawnRams:= 0
    MaxTotalPawns:= 0

    '// Now, perform the analysis
    if player==WHITE'( fromWhosePerspective == jcPlayer.SIDE_WHITE )
    
      repeat i from 0 to 7'for( int i = 0; i < 8; i++ )
        MaxMostAdvanced[ i ]:= 63
        MinMostBackward[ i ]:= 63
        MaxPassedPawns[ i ]:= 63

      repeat square from 55 to 8 step -1'= 55; square >= 8; square-- )
      
        '// Look for a white pawn first, and count its properties
        if ( GetPiece( square, WHITE ) == WHITE_PAWN )
        
          '// What is the pawn's position, in rank-file terms?
          rank:= square >> 3
          afile:= square // 8

          '// This pawn is now the most advanced of all white pawns on its file
          MaxPawnFileBins[ afile ]++
          MaxTotalPawns++
          MaxMostAdvanced[ afile ]:= square

          '// Is this pawn on a white or a black square?
          if ( ( rank // 2 ) == ( afile // 2 ) )
            MaxPawnColorBins[ 0 ]++
          else
            MaxPawnColorBins[ 1 ]++

          '// Look for a "pawn ram", i.e., a situation where a black pawn
          '// is located in the square immediately ahead of this one.
          if (GetPiece( square - 8, BLACK ) == BLACK_PAWN )
            PawnRams++
        
        '// Now, look for a BLACK pawn
        elseif (GetPiece( square,BLACK ) == BLACK_PAWN )
        
          '// If the black pawn exists, it is the most backward found so far
          '// on its file
          afile:= square // 8
          MinPawnFileBins[ afile ]++
          MinMostBackward[ afile ]:= square


    else '// Analyze from Black's perspective
    
      repeat i from 0 to 7'= 0; i < 8; i++ )
        MaxMostAdvanced[ i ]:= 0
        MaxPassedPawns[ i ]:= 0
        MinMostBackward[ i ]:= 0
      repeat square from 8 to 55'for( int square = 8; square < 56; square++ )
      
        if ( GetPiece( square,BLACK ) == BLACK_PAWN )
        
          '// What is the pawn's position, in rank-file terms?
          rank:= square >> 3
          afile:= square // 8

          '// This pawn is now the most advanced of all white pawns on its file
          MaxPawnFileBins[ afile ]++
          MaxTotalPawns++
          MaxMostAdvanced[ afile ]:= square

          if ( ( rank // 2 ) == ( afile // 2 ) )
            MaxPawnColorBins[ 0 ]++
          else
            MaxPawnColorBins[ 1 ]++

          if ( GetPiece( square + 8,WHITE ) == WHITE_PAWN )
            PawnRams++
        
        elseif ( GetPiece( square,WHITE ) == WHITE_PAWN )
          afile:= square // 8
          MinPawnFileBins[ afile ]++
          MinMostBackward[ afile ]:= square


PRI EvalMaterial:value|mTotal,matDiff,val
  ' If both sides are equal, no need to compute anything!
  if  MaterialValue[ BLACK ] == MaterialValue[ WHITE ] 
    return 0

  ser.dec(MaterialValue[ BLACK ] )
  ser.tx(32)
  ser.dec(MaterialValue[ WHITE ] )
  ser.tx(32)
  
  
  mTotal:= MaterialValue[ BLACK ] + MaterialValue[ WHITE ]

  if MaterialValue[ BLACK ] >  MaterialValue[ WHITE ]
    'Black in lead
    matDiff:= MaterialValue[ BLACK ] - MaterialValue[WHITE ]
    val:= 2400 <# matDiff
    val+= ( matDiff * ( 12000 - mTotal ) * NumPawns[BLACK ] ) / ( 6400 * ( NumPawns[BLACK ] + 1 ) )
      if Player == BLACK 
        return val
      else
        return -val 
  else
    'White in lead
    matDiff:= MaterialValue[WHITE] - MaterialValue[BLACK]
    val:= 2400 <# matDiff
    val+= ( matDiff * ( 12000 - mTotal ) * NumPawns[WHITE] ) / ( 6400 * ( NumPawns[WHITE] + 1 ) )
      if Player == WHITE 
        return val
      else
        return -val     


  

PUB PickBestMove|ThePlayer, bestSoFar,currentAlpha,movScore, i, pMove, pString,nMoves, depth
  ThePlayer:=player 'record who's turn it really is

  depth:=1'0'AlphaBetaDepth
  
    bestSoFar:= ALPHABETA_MINVAL
    'jcBoard newBoard = new jcBoard();
    'jcMove mov;
    currentAlpha:= ALPHABETA_MINVAL

  'Human's Applymove has already generated a list of moves of the current player.
  'Do it again anyway
   ComputeLegalMoves
  'Backup the move list for iteration 
  bytemove (@LegalMoveListBackup,@LegalMoveList,MoveListSize)

  'back up the original starting point board?
  longmove(@GameStateBackup,@GameState,GameStateSize)
  Player:=GameState.byte[0] 
     
  nMoves:=nLegalMoves
  'Iterate backup move list until we find a valid one
  repeat i from 0 to nMoves-1

    'get next move
    pMove:=@LegalMoveListBackup+i*MOVE_Bytes
    
    'apply move
    if ApplyMove2(pMove) 
      'This was a legal move!
      'undo change player
      player:=ThePlayer
      'score the move
      ser.str(string("AlphaBeta"))
      ser.dec(i)
      ser.str(string("of"))
      ser.dec(nMoves-1)
      ser.tx(13)
      
      movScore:=AlphaBeta(MINNODE,depth,currentAlpha,ALPHABETA_MAXVAL)      
    else
      'Restore original board
      longmove(@GameState,@GameStateBackup,GameStateSize)
      Player:=GameState.byte[0] 
      next
    currentAlpha#>=movScore 'set current alpha to maximum of current value and movScore
    if (movScore>bestSoFar)
       bytemove (@CurrentMove,pMove,MOVE_Bytes) 'store best move
       bestSoFar:=movScore
       'theMove.MoveEvaluation = movScore;




      'return 'test       

    'Restore original board
    longmove(@GameState,@GameStateBackup,GameStateSize)
    Player:=GameState.byte[0] 

    'bytemove (@CurrentMove,pMove, MOVE_Bytes) 'set best move
          'return 'test
    'return 'test

  '// And now, if the best we can do is ALPHABETA_GIVEUP or worse, then it is
  '// time to resign...  Unless the opponent was kind wnough to put us in
  '// stalemate!
  if bestSoFar =< ALPHABETA_GIVEUP
    'Make sure not in Checkmate or Stalemate
    ComputeLegalMoves
    pString:=TestForCheck
    case byte[pString]
      "S": 'We're in stalemate
        byte[@CurrentMove][MOVE_Type]:= MOVE_STALEMATE
      "C": 'Either in checkmate or soon to be
        byte[@CurrentMove][MOVE_Type]:= MOVE_RESIGN
      
        
PRI AlphaBeta(nodeType, depth, alpha, beta)|bestSoFar,pMoves, nMoves, i, pMove, movScore, currentAlpha, currentBeta  'Basic AlphaBeta

  ' If we have reached the maximum depth of the search, stop recursion
  'ser.str(string("AlphaBeta",13))
  ' and begin quiescence search
  if depth == 0
    'return QuiescenceSearch( nodeType, alpha, beta ) 
    return EvaluateComplete

  'need to back-up and restore board at each depth

   'Backup board for this depth
  longmove(@AlphaBetaGameStateBackup[(depth-1)*GameStateSize],@GameState,GameStateSize)  
    
  if !ComputeLegalMoves
    return ALPHABETA_ILLEGAL

  'Backup the move list for iteration
  pMoves:=@AlphaBetaLegalMoveList[MoveListSize*(depth-1)]
  bytemove (pMoves,@LegalMoveList,MoveListSize)
  nMoves:=nLegalMoves 

  if nodeType==MAXNODE
    'Case #1: We are searching a Max Node
    bestSoFar:= ALPHABETA_MINVAL
    currentAlpha:= alpha    
    'Iterate backup move list until we find a valid one
    repeat i from 0 to nMoves-1
      'get next move
      pMove:=pMoves+i*MOVE_Bytes
      'apply move
      if ApplyMove2(pMove)
        'This was a legal move!
         'restore player
         Player^=1 
         movScore:=AlphaBeta( !nodeType, depth - 1, currentAlpha, beta )
      
      else  ' Ignore illegal moves in the alphabeta evaluation 
        'Restore board
        longmove(@GameState,@AlphaBetaGameStateBackup[(depth-1)*GameStateSize],GameStateSize) 
        next
      currentAlpha#>=movScore 'set current alpha to maximum of current value and movScore

      'Is the current successor better than the previous best?
      if  movScore > bestSoFar 
        bestSoFar:= movScore
        'Can we cutoff now?
        if  bestSoFar => beta 
            return bestSoFar
            
      ' Test for checkmate or stalemate
      ' Both cases occur if and only if there is no legal move for MAX, i.e.,
      ' if "bestSoFar" is ALPHABETA_MINVAL.  There are two cases: we
      ' have checkmate (in which case the score is accurate) or stalemate (in
      ' which case the position should be re-scored as a draw with value 0.
      if  bestSoFar =< ALPHABETA_MINVAL 
        ' Can MIN capture MAX's king?  First, ask the machine to generate
        ' moves for MIN

        'Change Player
        GameState.byte[0]^=1
        Player:=GameState.byte[0]
        
        ' And if one of MIN's moves is a king capture, indicating that the
        ' position is illegal, we have checkmate and must return MINVAL.  We
        ' add the depth simply to "favor" delaying tactics: a mate in 5 will
        ' score higher than a mate in 3, because the likelihood that the
        ' opponent will miss it is higher; might as well make life difficult!
        if  !ComputeLegalMoves
          return bestSoFar + depth
        else
          return 0

     'Restore board
      longmove(@GameState,@AlphaBetaGameStateBackup[(depth-1)*GameStateSize],GameStateSize)         

  else  'Case #2: Min Node
    bestSoFar:= ALPHABETA_MAXVAL
    currentBeta:= beta
    'Iterate backup move list until we find a valid one
    repeat i from 0 to nMoves-1
      'get next move
      pMove:=pMoves+i*MOVE_Bytes
      'apply move
      if ApplyMove2(pMove)
        'restore player
        Player^=1 
        movScore:=AlphaBeta( !nodeType, depth - 1, alpha, currentBeta )
      
      else ' Ignore illegal moves in the alphabeta evaluation
        'Restore board
        longmove(@GameState,@AlphaBetaGameStateBackup[(depth-1)*GameStateSize],GameStateSize) 
        next
      currentBeta<#=movScore 'set current beta to minimum of current value and movScore

      'Is the current successor better than the previous best?
      if  movScore < bestSoFar 
        bestSoFar:= movScore
        'Can we cutoff now?
        if  bestSoFar =< alpha 
            return bestSoFar

      'Check for mates      
      if  bestSoFar => ALPHABETA_MAXVAL 

        'Change Player
        GameState.byte[0]^=1
        Player:=GameState.byte[0]
        
        if  !ComputeLegalMoves
          return bestSoFar + depth
        else
          return 0      
            
  ' If we haven't returned yet, we have found an accurate minimax score
  ' for a position which is neither a checkmate nor a stalemate
  return bestSoFar

      
     

PUB MakeMove(StartSquare, EndSquare):pErrorString|pMove
  pMove:=@CurrentMove
  byte[pMove][MOVE_SourceSquare]:=StartSquare
  byte[pMove][MOVE_DestinationSquare]:=EndSquare
  byte[pMove][MOVE_Type]:= MOVE_NORMAL

  '// Time to try to figure out what the move means!
  'if ( Player==WHITE ) 'combining black&white logic
    ' Is there a piece (of the moving player) on SourceSquare?
    ' If not, abort
    byte[pMove][MOVE_MovedPiece]:= GetPiece(StartSquare,White+Player)
    if ( byte[pMove][MOVE_MovedPiece] == NoPiece )
      return String(" doesn't have a piece on that square")
   
    'Three cases: there is a piece on the destination square (a capture),
    'the destination square allows an en passant capture, or it is a
    'simple non-capture move.  If the destination contains a piece of the
    'moving side, abort
    if ( GetPiece(EndSquare,White+Player) <> NoPiece )
      return String(" can't capture it's own piece!" )

   
  'Pawn Promotion?
  if (((byte[pMove][MOVE_MovedPiece] == WHITE_PAWN) and (byte[pMove][MOVE_DestinationSquare] < 8)) or ((Byte[pMove][MOVE_MovedPiece] == BLACK_PAWN) and (byte[pMove][MOVE_DestinationSquare] > 55)))
    byte[pMove][MOVE_Type] += MOVE_PROMOTION_QUEEN
    return String(" pawn promotion.")  'need to get exact promotion choice from user...

  'Just a regular move
  return 0  


PRI BitOnSquare(Piece, Square):bOnSquare
  'check given bitboard for a set bit on given square
  if square<32
    return BitBoard[Piece<<1] | (1<<Square)
  else
    return BitBoard[Piece<<1+1] | (1<<(Square-32))  

PUB GetEnPassantSquare:EPSquare
  'for debugging
  return GameState.byte[3]

PUB GetPlayer:iPlayer
  return Player  

PUB ApplyMove:bValidBoard|pMove,pLegalMove, i
  debug:=0
  pMove:=@CurrentMove

  'See if the current board is valid and make a list of possible moves
  if not ComputeLegalMoves
    debug:=675
    return false 'Board is invalid

  'now, See if move is in move list
  repeat i from 0 to nLegalMoves-1
    pLegalMove:=@LegalMoveList+i*MOVE_Bytes
    if (byte[pMove][MOVE_SourceSquare]==byte[pLegalMove][MOVE_SourceSquare]) AND (byte[pMove][MOVE_DestinationSquare]==byte[pLegalMove][MOVE_DestinationSquare])
      'Move is in the Quasi-Legal list, but may still leave the King in check (or try to castle through check)
      ' So, make a backup of the board in case we need to take the move back
      longmove(@GameStateBackup,@GameState,GameStateSize)

      if not ApplyMove2(pLegalMove)
        'New board is illegal!  Must have left a king in check...
        'revert
        longmove(@GameState,@GameStateBackup,GameStateSize)
        Player:=GameState.byte[0]
        return false
        
      return true  'found the move on the list and it doesn't leave king in check or castle through check      

  return false 'Move is not on legal list!


PRI ApplyMove2(pLegalMove)|ExtraKings
  'Actually apply a move known to be at lease psuedovalid
  'move the piece
  debug:=9876
  RemovePiece(byte[pLegalMove][MOVE_SourceSquare] )
  'Adjust for captures
  if (byte[pLegalMove][MOVE_Type]&MOVE_CAPTURE)
    'this is a capture, remove the piece from destination square
    'Unless EnPassant move
    if (byte[pLegalMove][MOVE_Type]&MOVE_EN_PASSANT)
      'EnPassant, calculate location of captured pawn and remove it
      if Player==White
         RemovePiece(GameState.byte[3]+8)
         'repeat
      else
         RemovePiece(GameState.byte[3]-8)
         debug:=4567
         'repeat
         
    else
      'normal capture
      RemovePiece(byte[pLegalMove][MOVE_DestinationSquare])
  'put peice in new position
  case byte[pLegalMove][MOVE_Type]&PROMOTION_MASK
    0: 'No pawn promotion
      AddPiece(byte[pLegalMove][MOVE_DestinationSquare],byte[pLegalMove][MOVE_MovedPiece] )
    MOVE_PROMOTION_QUEEN:
      'promote to queen
      AddPiece(byte[pLegalMove][MOVE_DestinationSquare],White_Queen+Player )  
    MOVE_PROMOTION_ROOK:
      'promote to queen
      AddPiece(byte[pLegalMove][MOVE_DestinationSquare],White_Rook+Player )                 
    MOVE_PROMOTION_BISHOP:
      'promote to queen
      AddPiece(byte[pLegalMove][MOVE_DestinationSquare],White_Bishop+Player ) 
    MOVE_PROMOTION_KNIGHT:
      'promote to queen
      AddPiece(byte[pLegalMove][MOVE_DestinationSquare],White_Knight+Player )
      
    
  'If this is a pawn double move, set the EnPassant Square
  case byte[pLegalMove][MOVE_MovedPiece]
    White_Pawn:
      if (byte[pLegalMove][MOVE_SourceSquare]- byte[pLegalMove][MOVE_DestinationSquare])==16
        'Set EnPassant to inbetween source and dest
        GameState.byte[3]:=byte[pLegalMove][MOVE_SourceSquare]-8
      else
        GameState.byte[3]:=-1
    Black_Pawn:
      if (byte[pLegalMove][MOVE_SourceSquare]- byte[pLegalMove][MOVE_DestinationSquare])==-16
        'Set EnPassant to inbetween source and dest
        GameState.byte[3]:=byte[pLegalMove][MOVE_SourceSquare]+8
      else
        GameState.byte[3]:=-1
    Other:
      'Clear the EnPassant
      GameState.byte[3]:=-1
   
  'Adjust for Castling
  ExtraKings:=0
  if (byte[pLegalMove][MOVE_Type]&MOVE_CASTLING_KINGSIDE)
    'Add in the Extra Kings and move the Rook
    if Player==White
      ExtraKings:=Extrakings_White_Kingside
      BitBoard[White_King<<1+1]+=ExtraKings
      RemovePiece(63)
      AddPiece(61,White_Rook)         
    else
      ExtraKings:=Extrakings_Black_Kingside
      BitBoard[Black_King<<1]+=ExtraKings
      RemovePiece(7)
      AddPiece(5,Black_Rook)          
  elseif (byte[pLegalMove][MOVE_Type]&MOVE_CASTLING_QueenSIDE)    
    'Add in the Extra Kings and move the Rook
    if Player==White
      ExtraKings:=Extrakings_White_Queenside
      BitBoard[White_King<<1+1]+=ExtraKings
      RemovePiece(56)
      AddPiece(59,White_Rook)         
    else
      ExtraKings:=Extrakings_Black_Kingside
      BitBoard[Black_King<<1]+=ExtraKings
      RemovePiece(0)
      AddPiece(3,Black_Rook)
   
  'Adjust Castling Flags
  'If a king moves then can't castle
  if byte[pLegalMove][MOVE_MovedPiece]==White_King
    GameState.byte[1]&=$FFFF_FFFC  'White can't castle
  if byte[pLegalMove][MOVE_MovedPiece]==Black_King
    GameState.byte[2]&=$FFFF_FFFC  'Black can't castle
  'if anything moves from corners then castling there is impossible
  case byte[pLegalMove][MOVE_SourceSquare]
    0:  GameState.byte[2]&=$FFFF_FFFE  'Black can't castle queenside
    7:  GameState.byte[2]&=$FFFF_FFFD  'Black can't castle kingside
    56:  GameState.byte[1]&=$FFFF_FFFE  'White can't castle queenside
    63:  GameState.byte[1]&=$FFFF_FFFD  'White can't castle kingside      


  'Change Player
  GameState.byte[0]^=1
  Player:=GameState.byte[0]
   
  'Test to make sure new board is legal
  if not ComputeLegalMoves
    return false
   
  'remove any extra kings
  if ExtraKings
    if Player==Black
      BitBoard[White_King<<1+1]-=ExtraKings
    else
      BitBoard[Black_King<<1]-=ExtraKings

  return true          

'0  1  2  3  4  5  6  7
'8  9  10 11 12 13 14 15
'16 17 18 19 20 21 22 23
'24 25 26 27 28 29 30 31
'32 33 34 35 36 37 38 39
'40 41 42 43 44 45 46 47
'48 49 50 51 52 53 54 55
'56 57 58 59 60 61 62 63

PUB TestForCheck:sMessage|pMove,pLegalMove, i, NewPlayer, bCheck, bValidMove, n
  'Applymove has already generated a list of moves of the current player.
  'Do it again anyway
   'ComputeLegalMoves
  'Now, see if any of them are valid to make sure game isn't over
  'Backup the move list 
  bytemove (@LegalMoveListBackup,@LegalMoveList,MoveListSize)



  'Iterate backup move list until we find a valid one
  bValidMove:=false
  n:= nLegalMoves-1
  repeat i from 0 to n
    'get next move
    pLegalMove:=@LegalMoveListBackup+i*MOVE_Bytes
    bytemove (@CurrentMove,pLegalMove,MOVE_Bytes)
    'apply move
    if ApplyMove
      'This was a legal move!
      'Restore board
      longmove(@GameState,@GameStateBackup,GameStateSize)

      'Player^=1 'change player back

      'Now, look to see if it puts the opponent's King in Check  
      'Do this by seeing if the current board is valid if the player that just moved can move again...
      'Player^=1 'change player for a second    
      bCheck:=ComputeLegalMoves
      Player^=1 'change player back
      if not bCheck
        'Last move put opposing King in check
        return string("Check    ")
      return string("         ")

  'If still here, then there are no legal moves!
  'Now, look to see if it puts the opponent's King in Check  
  'Do this by seeing if the current board is valid if the player that just moved can move again...
  Player^=1 'change for a second    
  bCheck:=ComputeLegalMoves
  Player^=1 'change player back
  if not bCheck
    'Last move put opposing King in check
    return string("Checkmate")
  return string("Stalemate")      
       

PUB GetPlayerString:pStr
  if GameState.byte[0]
    return String("Black")
  else
    return String("White")
  

PUB GetDebug
  return debug 'nLegalMoves

PRI ComputeLegalMoves:bValidBoard
  'returns false if any move can capture the king
  'Clear previous move List (by zeroing nLegalMoves)
  nLegalMoves:=0

  debug:=1000
  
  'Clean up castling constructs
  'ClearExtraKeys

  'AddToMoveList(48,40,0)

  'In the future, should create a temporary bitboard so that we can remove each found piece and test if all pieces are gone
  'Or, count # piece in bitboard and decrement each time found one...

  if not ComputePawnMoves
    return false

  if not ComputeQueenMoves
    return false

  if not ComputeBishopMoves(false)
    return false

  if not ComputeRookMoves(false)
    return false

  if not ComputeKnightMoves
    return false

  if not ComputeKingMoves
    return false
          
  return true

PRI ComputeQueenMoves:bValidBoard
  'Note:  There can be more than one queen
  if not ComputeBishopMoves(true)
    return false
  if not ComputeRookMoves(true)
    return false
    
  return true

PRI ComputePawnMoves:bValidBoard|piece, square, dest, pPiecesBitboard, bBlack
  piece:=Pawn+Player
  bBlack:=piece&Black
  'calculate address of piece bitboard
  pPiecesBitboard:=@Bitboard[piece<<1]
  'if bitboard empty then just return
  if long[pPiecesBitboard][0]==0 and long[pPiecesBitboard][1]==0
    return true

  'debug:=1
  'this one is complex, so break into black and white sections
  if piece==White_Pawn
    'White Pawn
    'Scan bitboard for piece
    repeat Square from 0 to 55
      if not bPieceOnSquareBitboard(pPiecesBitboard,square)
        'debug:=99
        next 'no piece on this square
      'now look for moves
      'First, look at a regular move
      'debug:=2
      dest:=square-8
      'if square>15
        'no promotion
      if not ( bPieceOnSquare(White_AllPieces, dest) or  bPieceOnSquare(Black_AllPieces, dest) )
        'dest square is empty so add to list
        AddToMoveList(square, dest, piece)
        'debug:=2
        'Maybe do a double move if on first row
        if square=>48
          dest-=8
          if not ( bPieceOnSquare(White_AllPieces, dest) or  bPieceOnSquare(Black_AllPieces, dest) )
            'debug:=3
            'dest square is empty so add to list
            AddToMoveList(square, dest, piece)

      'Now, look at captures...
      'Capture right
      if (square//8)<>7
        dest:=square-7
        if bPieceOnSquare(Black_AllPieces, dest)
          'add this capture to list
          if not AddToMoveList(square, dest, piece)
            return false
        else
          'en Passant?
          if GameState.byte[3]==dest
          'add this capture to list
            AddToMoveList(square, dest, piece)
      'Capture left
      if (square//8)<>0
        dest:=square-9
        if bPieceOnSquare(Black_AllPieces, dest)
          'add this capture to list
          if not AddToMoveList(square, dest, piece)
            return false
        else      
          'en Passant?
          if GameState.byte[3]==dest 'bPieceOnSquare(Black_EnPassant, dest)
          'add this capture to list
            AddToMoveList(square, dest, piece)          

  else
    'Black Pawn
    'Scan bitboard for piece
    repeat Square from 8 to 63
      if not bPieceOnSquareBitboard(pPiecesBitboard,square)
        next 'no piece on this square
      'now look for moves
      'First, look at a regular move
      dest:=square+8
      if not ( bPieceOnSquare(White_AllPieces, dest) or  bPieceOnSquare(Black_AllPieces, dest) )
        'dest square is empty so add to list
        AddToMoveList(square, dest, piece)
        'Maybe do a double move if on first row
        if square=<15
          dest+=8
          if not ( bPieceOnSquare(White_AllPieces, dest) or  bPieceOnSquare(Black_AllPieces, dest) )
            AddToMoveList(square, dest, piece)

      'Now, look at captures...
      'piece:=Black_Pawn
      'Capture right
      if (square//8)<>7
        dest:=square+9
        if bPieceOnSquare(White_AllPieces, dest)
          'add this capture to list
          if not AddToMoveList(square, dest, piece)
            return false
        else
          'en Passant?
          if GameState.byte[3]==dest
          'add this capture to list
            AddToMoveList(square, dest, Black_Pawn)'piece)
            'repeat
      'Capture left
      if (square//8)<>0
        dest:=square+7
        if bPieceOnSquare(White_AllPieces, dest)
          'add this capture to list
          if not AddToMoveList(square, dest, piece)
            return false
        else      
          'en Passant?
          if GameState.byte[3]==dest 'bPieceOnSquare(Black_EnPassant, dest)
          'add this capture to list
            AddToMoveList(square, dest, Black_Pawn)'piece)
            'repeat

                
  return true   
'0  1  2  3  4  5  6  7
'8  9  10 11 12 13 14 15
'16 17 18 19 20 21 22 23
'24 25 26 27 28 29 30 31
'32 33 34 35 36 37 38 39
'40 41 42 43 44 45 46 47
'48 49 50 51 52 53 54 55
'56 57 58 59 60 61 62 63

PRI ComputeRookMoves(bQueen):bValidBoard|piece, pPiecesBitboard, square, ray, dest
  'computes moves for Rook (or queen if passed true)
  'figure out which piece this is
  if bQueen
    piece:=Queen+Player
  else
    piece:=Rook+Player  
  'calculate address of piece bitboard
  pPiecesBitboard:=@Bitboard[piece<<1]
  'if bitboard empty then just return
  if long[pPiecesBitboard][0]==0 and long[pPiecesBitboard][1]==0
    return true

  'Scan bitboard for piece
  repeat square from 0 to 63
    if bPieceOnSquareBitboard(pPiecesBitboard,square)
      'found a piece on this square, so look for moves      
      'Ray#1 (N)
      dest:=square
      repeat while dest>7  'stay on board
        dest-=8
        if bPieceOnSquare(AllPieces+Player, dest)
          quit  'break this loop (terminate ray) if landed on own piece
        bValidBoard:= AddToMoveList(square, dest, piece) 
        if bValidBoard==false
          return false
        if CapturedPiece<>NoPiece
          'this was a capture, so terminate ray
          quit
      'Ray#2 (E)
      dest:=square
      repeat while ((dest//8)<7) 'stay on board
        dest+=1
        if bPieceOnSquare(AllPieces+Player, dest)
          quit  'break this loop (terminate ray) if landed on own piece
        bValidBoard:= AddToMoveList(square, dest, piece) 
        if bValidBoard==false
          return false
        if CapturedPiece<>NoPiece
          'this was a capture, so terminate ray
          quit              
      'Ray#3 (S)
      dest:=square
      repeat while (dest<56)   'stay on board
        dest+=8
        if bPieceOnSquare(AllPieces+Player, dest)
          quit  'break this loop (terminate ray) if landed on own piece
        bValidBoard:= AddToMoveList(square, dest, piece) 
        if bValidBoard==false  
          return false
        if CapturedPiece<>NoPiece
          'this was a capture, so terminate ray
          quit
      'Ray#4 (W)
      dest:=square
      repeat while  ((dest//8)>0)  'stay on board
        dest-=1
        if bPieceOnSquare(AllPieces+Player, dest)
          quit  'break this loop (terminate ray) if landed on own piece
        bValidBoard:= AddToMoveList(square, dest, piece) 
        if bValidBoard==false  
          return false
        if CapturedPiece<>NoPiece
          'this was a capture, so terminate ray
          quit
 
  'all done
  return true
                        
PRI ComputeBishopMoves(bQueen):bValidBoard|piece, pPiecesBitboard, square, ray, dest
  'computes moves for bishop (or queen if passed true)
  'figure out which piece this is
  if bQueen
    piece:=Queen+Player
  else
    piece:=Bishop+Player  
  'calculate address of piece bitboard
  pPiecesBitboard:=@Bitboard[piece<<1]
  'if bitboard empty then just return
  if long[pPiecesBitboard][0]==0 and long[pPiecesBitboard][1]==0
    return true

  'Scan bitboard for piece
  repeat square from 0 to 63
    if bPieceOnSquareBitboard(pPiecesBitboard,square)
      'found a piece on this square, so look for moves      
      'Ray#1 (NE)
      dest:=square
      repeat while (dest>7) and ((dest//8)<7)  'stay on board
        dest-=7
        if bPieceOnSquare(AllPieces+Player, dest)
          quit  'break this loop (terminate ray) if landed on own piece
        debug:=897 
        bValidBoard:= AddToMoveList(square, dest, piece) 
        if bValidBoard==false
          return false
        if CapturedPiece<>NoPiece
          'this was a capture, so terminate ray
          quit
      'Ray#2 (SE)
      dest:=square
      repeat while (dest<56) and ((dest//8)<7)  'stay on board
        dest+=9
        if bPieceOnSquare(AllPieces+Player, dest)
          quit  'break this loop (terminate ray) if landed on own piece
        bValidBoard:= AddToMoveList(square, dest, piece) 
        if bValidBoard==false  
          return false
        if CapturedPiece<>NoPiece
          'this was a capture, so terminate ray
          quit        
      'Ray#3 (SW)
      dest:=square
      repeat while (dest<56) and ((dest//8)>0)  'stay on board
        dest+=7
        if bPieceOnSquare(AllPieces+Player, dest)
          quit  'break this loop (terminate ray) if landed on own piece
        bValidBoard:= AddToMoveList(square, dest, piece) 
        if bValidBoard==false  
          return false
        if CapturedPiece<>NoPiece
          'this was a capture, so terminate ray
          quit
      'Ray#4 (NW)
      dest:=square
      repeat while (dest>7) and ((dest//8)>0)  'stay on board
        dest-=9
        if bPieceOnSquare(AllPieces+Player, dest)
          quit  'break this loop (terminate ray) if landed on own piece
        bValidBoard:= AddToMoveList(square, dest, piece) 
        if bValidBoard==false  
          return false
        if CapturedPiece<>NoPiece
          'this was a capture, so terminate ray
          quit
  'all done
  return true

PRI ComputeKnightMoves:bValidBoard|piece, pPiecesBitboard, square, ray, dest,mod8
  'computes moves for knight
  piece:=Knight+Player
  'calculate address of piece bitboard
  pPiecesBitboard:=@Bitboard[piece<<1]
  'if bitboard empty then just return
  if long[pPiecesBitboard][0]==0 and long[pPiecesBitboard][1]==0
    return true

  'Scan bitboard for piece
  repeat square from 0 to 63
    if bPieceOnSquareBitboard(pPiecesBitboard,square)
      mod8:=square//8 'slight optimization
      'found a piece on this square, so look for moves      
      'Move#1 (ENE)
      if mod8<6 and square>7  'stay on board
        dest:=square-6
        if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
          bValidBoard:= AddToMoveList(square, dest, piece) 
          if bValidBoard==false
            return false
      'Move#2 (ESE)
      if mod8<6 and square<56  'stay on board    'stay on board
        dest:=square+10
        if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
          bValidBoard:= AddToMoveList(square, dest, piece) 
          if bValidBoard==false  
            return false
      'Move#3 (SSE)
      if mod8<7 and square<48  'stay on board    'stay on board
        dest:=square+17
        if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
          bValidBoard:= AddToMoveList(square, dest, piece) 
          if bValidBoard==false  
            return false
      'Move#4 (SSW)
      if mod8>0 and square<48  'stay on board    'stay on board
        dest:=square+15
        if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
          bValidBoard:= AddToMoveList(square, dest, piece) 
          if bValidBoard==false  
            return false
       'Move#5 (WSW)
      if mod8>1 and square<56  'stay on board    'stay on board
        dest:=square+6
        if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
          bValidBoard:= AddToMoveList(square, dest, piece) 
          if bValidBoard==false  
            return false
       'Move#6 (WNW)
      if mod8>1 and square>7  'stay on board    'stay on board
        dest:=square-10
        if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
          bValidBoard:= AddToMoveList(square, dest, piece) 
          if bValidBoard==false  
            return false
       'Move#7 (NNW)
      if mod8>0 and square>15  'stay on board    'stay on board
        dest:=square-17
        if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
          bValidBoard:= AddToMoveList(square, dest, piece) 
          if bValidBoard==false  
            return false            
       'Move#8 (NNE)
      if mod8<7 and square>15  'stay on board    'stay on board
        dest:=square-15
        if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
          bValidBoard:= AddToMoveList(square, dest, piece) 
          if bValidBoard==false  
            return false
  'all done
  return true          
PRI ComputeKingMoves:bValidBoard|piece, pPiecesBitboard, square, ray, dest,mod8
  'computes moves for the King
  piece:=King+Player
  'calculate address of piece bitboard
  pPiecesBitboard:=@Bitboard[piece<<1]

  'There must be one and only one King, so just find it directly
  if long[pPiecesBitboard][0]<>0
    square:=(>|long[pPiecesBitboard][0])-1         'Using the Bitwise encode command!
  else
    square:=(>|long[pPiecesBitboard][1])+31

  mod8:=square//8 'slight optimization
  'Move#1 (N)
  if square>7  'stay on board
    dest:=square-8
    if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
      bValidBoard:= AddToMoveList(square, dest, piece) 
      if bValidBoard==false
        return false
  'Move#2 (NE)
  if mod8<7 and square>7  'stay on board
    dest:=square-7
    if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
      bValidBoard:= AddToMoveList(square, dest, piece) 
      if bValidBoard==false
        return false
  'Move#3 (E)
  if mod8<7   'stay on board
    dest:=square+1
    if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
      bValidBoard:= AddToMoveList(square, dest, piece) 
      if bValidBoard==false
        return false
  'Move#4 (SE)
  if mod8<7 and square<56   'stay on board
    dest:=square+9
    if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
      bValidBoard:= AddToMoveList(square, dest, piece) 
      if bValidBoard==false
        return false
  'Move#5 (S)
  if square<56   'stay on board
    dest:=square+8
    if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
      bValidBoard:= AddToMoveList(square, dest, piece) 
      if bValidBoard==false
        return false                
  'Move#6 (SW)
  if mod8>0 and square<56   'stay on board
    dest:=square+7
    if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
      bValidBoard:= AddToMoveList(square, dest, piece) 
      if bValidBoard==false
        return false
  'Move#7 (W)
  if mod8>0    'stay on board
    dest:=square-1
    if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
      bValidBoard:= AddToMoveList(square, dest, piece) 
      if bValidBoard==false
        return false
  'Move#8 (NW)
  if mod8>0 and square>7    'stay on board
    dest:=square-9
    if not bPieceOnSquare(AllPieces+Player, dest)     'only if not landed on own piece
      bValidBoard:= AddToMoveList(square, dest, piece) 
      if bValidBoard==false
        return false

  'Now, look at White Castling
  if piece==White_King
    'White Kingside
    if (GameState.byte[1]&2)  'see if still allowed
      'Make sure empty squares between king and rook
      if  (BitBoard[White_AllPieces<<1+1]&EmptySquares_White_Kingside)==0 and (BitBoard[Black_AllPieces<<1+1]&EmptySquares_White_Kingside)==0
        'Castling looks valid.  Will make sure king doesn't pass through check using "ExtraKings" device after applying move
        AddToMoveList(60, 62, piece)         
    'White Queenside
    if (GameState.byte[1]&1)  'see if still allowed
      'Make sure empty squares between king and rook
      if not (BitBoard[White_AllPieces<<1+1]&EmptySquares_White_Queenside) and not(BitBoard[Black_AllPieces<<1+1]&EmptySquares_White_Queenside)
        'Castling looks valid.  Will make sure king doesn't pass through check using "ExtraKings" device after applying move
        AddToMoveList(60, 58, piece) 
  'Now, look at Black Castling
  if piece==Black_King
    'Black Kingside
    if (GameState.byte[2]&2)  'see if still allowed
      'Make sure empty squares between king and rook
      if  (BitBoard[Black_AllPieces<<1+1]&EmptySquares_Black_Kingside)==0 and (BitBoard[White_AllPieces<<1+1]&EmptySquares_Black_Kingside)==0
        'Castling looks valid.  Will make sure king doesn't pass through check using "ExtraKings" device after applying move
        AddToMoveList(4, 6, piece)         
    'Black Queenside
    if (GameState.byte[2]&1)  'see if still allowed
      'Make sure empty squares between king and rook
      if not (BitBoard[White_AllPieces<<1+1]&EmptySquares_Black_Queenside) and not(BitBoard[Black_AllPieces<<1+1]&EmptySquares_Black_Queenside)
        'Castling looks valid.  Will make sure king doesn't pass through check using "ExtraKings" device after applying move
        AddToMoveList(4, 2, piece)
        
  return true
                     
'0  1  2  3  4  5  6  7
'8  9  10 11 12 13 14 15
'16 17 18 19 20 21 22 23
'24 25 26 27 28 29 30 31
'32 33 34 35 36 37 38 39
'40 41 42 43 44 45 46 47
'48 49 50 51 52 53 54 55
'56 57 58 59 60 61 62 63
      
PRI AddToMoveList(square, dest, MovedPiece):bValidBoard |pMove, opponent, i, pMove2  'Create move structure and add to legal moves list
  'returns false(0) if captures a king, true(-1) otherwise
  CapturedPiece:=NoPiece
  
  opponent:=Black-player
  
  if (nLegalMoves++)==nMaxMoves
    'memory overflow
    return false 'really should trap this error better!

  pMove:=@LegalMoveList+(nLegalMoves-1)*MOVE_Bytes  

  byte[pMove][MOVE_SourceSquare]:=square
  byte[pMove][MOVE_DestinationSquare]:=dest
  byte[pMove][MOVE_MovedPiece]:=MovedPiece
  if bPieceOnSquare(AllPieces+opponent, dest)
    debug:=85
    'opponent is on this square, so it is a capture
    capturedPiece:=GetPiece(dest,opponent)
    byte[pMove][MOVE_CapturedPiece]:=capturedPiece
    byte[pMove][MOVE_Type]:=MOVE_CAPTURE
    if (capturedPiece==(KING+opponent))
      return false  'this board is invalid because we can capture the opponent's king!!!
  else
    byte[pMove][MOVE_Type]:=MOVE_NORMAL

  'See if this is a castling move
  if MovedPiece==White_King
    if square==60 and dest==62
      'White is castling Kingside
       byte[pMove][MOVE_Type]:=MOVE_CASTLING_KINGSIDE
    if square==60 and dest==58
      'White is castling Kingside
       byte[pMove][MOVE_Type]:=MOVE_CASTLING_QUEENSIDE
  if MovedPiece==Black_King
    if square==4 and dest==6
      'White is castling Kingside
       byte[pMove][MOVE_Type]:=MOVE_CASTLING_KINGSIDE
    if square==4 and dest==2
      'White is castling Kingside
       byte[pMove][MOVE_Type]:=MOVE_CASTLING_QUEENSIDE
         
  'Do some extra work for Pawns
  if MovedPiece==White_Pawn
    if dest<8     'is this a white pawn promotion? 
      'add a move for every type of promotion
      byte[pMove][MOVE_Type]+=MOVE_PROMOTION_QUEEN
      repeat i from 5 to 7 'log2 promotion type!
        if (nLegalMoves++)==nMaxMoves
          'memory overflow
          return false 'really should trap this error better!
        pMove2:=@LegalMoveList+(nLegalMoves-1)*MOVE_Bytes
        ByteMove(pMove2,pMove,Move_Bytes)
        pMove:=pMove2
        byte[pMove][MOVE_Type]&=NO_PROMOTION_MASK+1<<i  'change promotion type
    else
      if (GameState.byte[3]==dest) and  (capturedPiece==NoPiece)     'bPieceOnSquare(Black_EnPassant, dest)
        'this is an En Passant move
        capturedPiece:=Black_Pawn 
        byte[pMove][MOVE_CapturedPiece]:=Black_Pawn
        byte[pMove][MOVE_Type]:=MOVE_CAPTURE+MOVE_EN_PASSANT

        
  if MovedPiece==Black_Pawn
    if dest>55     'is this a black pawn promotion? 
      'add a move for every type of promotion
      byte[pMove][MOVE_Type]+=MOVE_PROMOTION_QUEEN
      repeat i from 5 to 7 'log2 promotion type!
        if (nLegalMoves++)==nMaxMoves
          'memory overflow
          return false 'really should trap this error better!
        pMove2:=@LegalMoveList+(nLegalMoves-1)*MOVE_Bytes
        ByteMove(pMove2,pMove,Move_Bytes)
        pMove:=pMove2
        byte[pMove][MOVE_Type]&=NO_PROMOTION_MASK+1<<i  'change promotion type
    else
      if ((GameState.byte[3])==dest) and  (capturedPiece==NoPiece)     'bPieceOnSquare(Black_EnPassant, dest)
        'this is an En Passant move
        capturedPiece:=White_Pawn 
        byte[pMove][MOVE_CapturedPiece]:=White_Pawn
        byte[pMove][MOVE_Type]:=MOVE_CAPTURE+MOVE_EN_PASSANT
      
      
  return true


        
'0  1  2  3  4  5  6  7
'8  9  10 11 12 13 14 15
'16 17 18 19 20 21 22 23
'24 25 26 27 28 29 30 31
'32 33 34 35 36 37 38 39
'40 41 42 43 44 45 46 47
'48 49 50 51 52 53 54 55
'56 57 58 59 60 61 62 63          

  

PRI AddPiece(Square, Piece): bAdded|SquareBits0, SquareBits1, Color
  'init
  SquareBits0:=0
  SquareBits1:=0
  Color:=Piece//2
   
  'Add a piece to a bitboard
  if square<32
    SquareBits0:= 1<<Square
    BitBoard[Piece<<1] |= SquareBits0
  else
    SquareBits1:= 1<<(Square-32) 
    BitBoard[Piece<<1+1] |= SquareBits1
   
  ' And note the new piece position in the bitboard containing all
  ' pieces of its color.  
  BitBoard[(White_AllPieces + Color)<<1] |= SquareBits0
  BitBoard[(White_AllPieces + Color)<<1+1] |= SquareBits1
   
   
  ' And adjust material balance accordingly
  MaterialValue[Color] += 10*PieceValue[Piece]
  if ( Piece == WHITE_PAWN )
    NumPawns[White]++
  elseif ( Piece == BLACK_PAWN )
    NumPawns[Black]++

PRI RemovePiece(Square): bAdded|SquareBits0, SquareBits1, Color, piece
  'find piece
  piece:=GetPiece(Square,Black)
  if piece==NoPiece
    piece:=GetPiece(Square,White)
    if piece==NoPiece
      return false  
  'init
  SquareBits0:=0
  SquareBits1:=0
  Color:=Piece//2
   
  'Add a piece to a bitboard
  if square<32
    SquareBits0:= 1<<Square
    BitBoard[Piece<<1] ^= SquareBits0
  else
    SquareBits1:= 1<<(Square-32) 
    BitBoard[Piece<<1+1] ^= SquareBits1
   
  ' And note the new piece position in the bitboard containing all
  ' pieces of its color.  
  BitBoard[(White_AllPieces + Color)<<1] ^= SquareBits0
  BitBoard[(White_AllPieces + Color)<<1+1] ^= SquareBits1
   
   
  ' And adjust material balance accordingly
  MaterialValue[Color] -= 10*PieceValue[Piece]
  if ( Piece == WHITE_PAWN )
    NumPawns[White]--
  elseif ( Piece == BLACK_PAWN )
    NumPawns[Black]--

  return true

PRI bPieceOnSquare(Piece, Square)|SquareBits
  'Look for piece on it's bitboard bitboard
  'Low long stored first in bitboard array...
  if square<32
    SquareBits:= 1<<Square
    if BitBoard[Piece<<1] & SquareBits
      return true
  else
    SquareBits:= 1<<(Square-32) 
    if BitBoard[Piece<<1+1] & SquareBits
      return true
  return false

PRI bPieceOnSquareBitboard(pBitboard,Square)|SquareBits
  'look for piece on a square given its bitboard
  if square<32
    SquareBits:= 1<<Square
    if long[pBitboard][0] & SquareBits
      return true
  else
    SquareBits:= 1<<(Square-32) 
    if long[pBitboard][1] & SquareBits
      return true
  return false    

PUB GetSquareBitmap(square):BmpAddress|p
'Get the bitmap that should be painted on given square
  p:=pBlank-user_charbase+@uchar '@Blank  
  case GetPiece(square,White)
    WHITE_PAWN:
      p:=@WhitePawn 
    WHITE_KNIGHT:
      p:=@WhiteKnight
    WHITE_BISHOP:
      p:=@WhiteBishop
    WHITE_ROOK:
      p:=@WhiteRook  
    WHITE_QUEEN:
      p:=@WhiteQueen
    WHITE_KING:
      p:=@WhiteKing

  case GetPiece(square,Black)
    BLACK_PAWN:
      p:=@BlackPawn 
    BLACK_KNIGHT:
      p:=@BlackKnight
    BLACK_BISHOP:
      p:=@BlackBishop
    BLACK_ROOK:
      p:=@BlackRook  
    BLACK_QUEEN:
      p:=@BlackQueen
    BLACK_KING:
      p:=@BlackKing

  return BmpAddress:=p+user_charbase-@uchar

PUB GetBorderBitmap(n):BmpAddress|p
  case n
    0: p:=@border_top
    1: p:=@border_right
    2: p:=@border_bottom
    3: p:=@border_left
    4: p:=@border_top_left
    5: p:=@border_top_right
    6: p:=@border_bottom_right
    7: p:=@border_bottom_left

  return BmpAddress:=p+user_charbase-@uchar  

PRI GetPiece(square,color)
'Scan bitboards for a piece of given color on this square
  if bPieceOnSquare(WHITE_KING+color, square) 
    return WHITE_KING+color
  if bPieceOnSquare(WHITE_QUEEN+color, square)
    return WHITE_QUEEN+color
  if bPieceOnSquare(WHITE_ROOK+color, square)
    return WHITE_ROOK+color
  if bPieceOnSquare(WHITE_KNIGHT+color, square)
    return WHITE_KNIGHT+color
  if bPieceOnSquare(WHITE_BISHOP+color, square)
    return WHITE_BISHOP+color
  if bPieceOnSquare(WHITE_PAWN+color, square)
    return WHITE_PAWN+color
  return NoPiece

DAT 'initialized data
  PieceValue byte 10,10,30,30,35,35,50,50,90,90,200,200   ' Numerical evaluation of piece material values

pBlank  word $0840  'address of the VGA_tile driver that will be used for blank squares
  
'DAT
padding LONG  7[16] 'alignment padding for the following user defined characters     

uchar long
'pieces are 2-bit bitmaps.  There are 13 types, each with nine 16x16 tiles. Size is 13*9*(256*2/8)  =7.5kB
WhitePawn long
        file "pawn_white3.dat"  '3x3=9
BlackPawn long
        file "pawn_black3.dat"  '3x3=9
WhiteRook long
        file "rook_white3.dat"  '3x3=9       
BlackRook long
        file "rook_black3.dat"  '3x3=9
WhiteKnight long
        file "Knight_white3.dat"  '3x3=9       
BlackKnight long
        file "Knight_black3.dat"  '3x3=9
WhiteBishop long
        file "Bishop_white3.dat"  '3x3=9       
BlackBishop long
        file "Bishop_black3.dat"  '3x3=9
WhiteQueen long
        file "Queen_white3.dat"  '3x3=9       
BlackQueen long
        file "Queen_black3.dat"  '3x3=9
WhiteKing long
        file "King_white3.dat"  '3x3=9       
BlackKing long
        file "King_black3.dat"  '3x3=9
'Blank long 0[144]  '9
Border_top long
        file "border_top.dat"  '1x1=1
Border_right long
        file "border_right.dat" '1x1=1
border_bottom long
        file "border_bottom.dat" '1x1
border_left long
        file "border_left.dat"  '1x1

Border_top_left long
        file "border_top_left.dat"  '1x1=1
Border_top_right long
        file "border_top_right.dat" '1x1=1
border_bottom_right long
        file "border_bottom_right.dat" '1x1
border_bottom_left long
        file "border_bottom_left.dat"  '1x1








        
